'use client'

import { useEffect, useState } from 'react'
import { useParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cliente, Suscripcion, Pago, Recordatorio } from '@/lib/types'
import Link from 'next/link'
import { format } from 'date-fns'

export default function ClienteDetailPage() {
  const params = useParams()
  const [cliente, setCliente] = useState<Cliente | null>(null)
  const [suscripciones, setSuscripciones] = useState<Suscripcion[]>([])
  const [pagos, setPagos] = useState<Pago[]>([])
  const [recordatorios, setRecordatorios] = useState<Recordatorio[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (params.id) {
      fetchCliente(params.id as string)
      fetchSuscripciones(params.id as string)
      fetchPagos(params.id as string)
      fetchRecordatorios(params.id as string)
    }
  }, [params.id])

  async function fetchCliente(id: string) {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('id', id)
        .single()

      if (error) throw error
      setCliente(data)
    } catch (error) {
      console.error('Error fetching cliente:', error)
    } finally {
      setLoading(false)
    }
  }

  async function fetchSuscripciones(clienteId: string) {
    try {
      const { data, error } = await supabase
        .from('suscripciones')
        .select('*')
        .eq('cliente_id', clienteId)
        .order('created_at', { ascending: false })

      if (error) throw error
      setSuscripciones(data || [])
    } catch (error) {
      console.error('Error fetching suscripciones:', error)
    }
  }

  async function fetchPagos(clienteId: string) {
    try {
      const { data, error } = await supabase
        .from('pagos')
        .select('*')
        .eq('cliente_id', clienteId)
        .order('fecha_pago', { ascending: false })
        .limit(10)

      if (error) throw error
      setPagos(data || [])
    } catch (error) {
      console.error('Error fetching pagos:', error)
    }
  }

  async function fetchRecordatorios(clienteId: string) {
    try {
      const { data, error } = await supabase
        .from('recordatorios')
        .select('*')
        .eq('cliente_id', clienteId)
        .eq('completado', false)
        .order('fecha_recordatorio', { ascending: true })
        .limit(5)

      if (error) throw error
      setRecordatorios(data || [])
    } catch (error) {
      console.error('Error fetching recordatorios:', error)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  if (!cliente) {
    return (
      <div className="text-center p-8">
        <p>Cliente no encontrado</p>
        <Link href="/clientes" className="text-blue-600 hover:underline">Volver a clientes</Link>
      </div>
    )
  }

  return (
    <div className="max-w-6xl mx-auto">
      <div className="mb-4">
        <Link href="/clientes" className="text-blue-600 hover:underline">← Volver a clientes</Link>
      </div>
      
      <div className="bg-white p-6 rounded-lg shadow mb-6">
        <div className="flex justify-between items-start mb-6">
          <div>
            <h1 className="text-3xl font-bold mb-2">
              {cliente.nombre || cliente.empresa || 'Cliente sin nombre'}
            </h1>
            {cliente.empresa && cliente.nombre && (
              <p className="text-gray-600 mb-2">{cliente.empresa}</p>
            )}
            <span className={`inline-block px-3 py-1 rounded text-sm ${
              cliente.activo ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
            }`}>
              {cliente.activo ? 'Activo' : 'Inactivo'}
            </span>
          </div>
          <div className="flex gap-2">
            <Link
              href={`/clientes/${cliente.id}/editar`}
              className="bg-gray-600 text-white px-4 py-2 rounded hover:bg-gray-700 transition text-sm flex items-center gap-2"
            >
              ✏️ Editar
            </Link>
            <Link
              href={`/suscripciones/nuevo?cliente_id=${cliente.id}`}
              className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition text-sm"
            >
              + Suscripción
            </Link>
            <Link
              href={`/pagos/nuevo?cliente_id=${cliente.id}`}
              className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 transition text-sm"
            >
              + Pago
            </Link>
            <Link
              href={`/recordatorios/nuevo?cliente_id=${cliente.id}`}
              className="bg-yellow-600 text-white px-4 py-2 rounded hover:bg-yellow-700 transition text-sm"
            >
              + Recordatorio
            </Link>
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {cliente.empresa && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Empresa</h3>
              <p className="text-gray-900">{cliente.empresa}</p>
            </div>
          )}
          {cliente.email && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Email</h3>
              <p className="text-gray-900">{cliente.email}</p>
            </div>
          )}
          {cliente.telefono && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Teléfono</h3>
              <p className="text-gray-900">{cliente.telefono}</p>
            </div>
          )}
          {cliente.rfc && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">RUT</h3>
              <p className="text-gray-900">{cliente.rfc}</p>
            </div>
          )}
          {cliente.direccion && (
            <div className="md:col-span-2">
              <h3 className="font-semibold text-gray-700 mb-1">Dirección</h3>
              <p className="text-gray-900">{cliente.direccion}</p>
            </div>
          )}
          {cliente.notas && (
            <div className="md:col-span-2">
              <h3 className="font-semibold text-gray-700 mb-1">Notas</h3>
              <p className="text-gray-900 whitespace-pre-wrap">{cliente.notas}</p>
            </div>
          )}
        </div>
      </div>

      {/* Suscripciones */}
      <div className="bg-white p-6 rounded-lg shadow mb-6">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-2xl font-bold">Suscripciones ({suscripciones.length})</h2>
          <Link
            href={`/suscripciones/nuevo?cliente_id=${cliente.id}`}
            className="text-blue-600 hover:underline text-sm"
          >
            Ver todas →
          </Link>
        </div>
        {suscripciones.length === 0 ? (
          <p className="text-gray-500">No hay suscripciones para este cliente.</p>
        ) : (
          <div className="space-y-3">
            {suscripciones.slice(0, 3).map((suscripcion) => {
              const vencida = suscripcion.fecha_fin && new Date(suscripcion.fecha_fin) < new Date()
              const porVencer = suscripcion.fecha_fin && (() => {
                const fin = new Date(suscripcion.fecha_fin)
                const hoy = new Date()
                const diasRestantes = Math.ceil((fin.getTime() - hoy.getTime()) / (1000 * 60 * 60 * 24))
                return diasRestantes <= 7 && diasRestantes > 0
              })()
              
              let estadoTexto: string = suscripcion.estado
              let estadoColor = 'bg-gray-100 text-gray-800'
              let borderColor = 'border-gray-200'
              
              if (vencida) {
                estadoTexto = 'Vencida'
                estadoColor = 'bg-red-100 text-red-800'
                borderColor = 'border-red-300 border-l-4'
              } else if (porVencer) {
                estadoTexto = 'Por vencer'
                estadoColor = 'bg-yellow-100 text-yellow-800'
                borderColor = 'border-yellow-300 border-l-4'
              } else if (suscripcion.estado === 'activa') {
                estadoTexto = 'En curso'
                estadoColor = 'bg-green-100 text-green-800'
                borderColor = 'border-green-300 border-l-4'
              }
              
              return (
                <Link
                  key={suscripcion.id}
                  href={`/suscripciones/${suscripcion.id}`}
                  className={`block border rounded p-3 hover:opacity-80 transition ${borderColor}`}
                >
                  <div className="flex justify-between items-center">
                    <div>
                      <p className="font-semibold">
                        {suscripcion.tipo === 'anual' ? '📅 Anual' : '📆 Mensual'} - ${suscripcion.precio.toLocaleString()}
                      </p>
                      <p className="text-sm text-gray-600">
                        {(() => {
                          const fechaInicio = suscripcion.fecha_inicio.split('T')[0]
                          const [year1, month1, day1] = fechaInicio.split('-')
                          const fechaFin = suscripcion.fecha_fin ? suscripcion.fecha_fin.split('T')[0] : null
                          if (fechaFin) {
                            const [year2, month2, day2] = fechaFin.split('-')
                            return `${day1}/${month1}/${year1} - ${day2}/${month2}/${year2}`
                          }
                          return `${day1}/${month1}/${year1} - Sin fecha fin`
                        })()}
                      </p>
                    </div>
                    <span className={`px-2 py-1 rounded text-xs font-medium ${estadoColor}`}>
                      {estadoTexto}
                    </span>
                  </div>
                </Link>
              )
            })}
          </div>
        )}
      </div>

      {/* Pagos */}
      <div className="bg-white p-6 rounded-lg shadow mb-6">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-2xl font-bold">Últimos Pagos ({pagos.length})</h2>
          <Link
            href={`/pagos?cliente_id=${cliente.id}`}
            className="text-blue-600 hover:underline text-sm"
          >
            Ver todos →
          </Link>
        </div>
        {pagos.length === 0 ? (
          <p className="text-gray-500">No hay pagos registrados para este cliente.</p>
        ) : (
          <div className="space-y-3">
            {pagos.map((pago) => {
              // Trabajar con strings YYYY-MM-DD directamente
              const fechaPagoStr = pago.fecha_pago.split('T')[0]
              const hoyStr = new Date().toISOString().split('T')[0]
              const vencido = fechaPagoStr < hoyStr
              
              const porVencer = (() => {
                const [year, month, day] = fechaPagoStr.split('-').map(Number)
                const [yearHoy, monthHoy, dayHoy] = hoyStr.split('-').map(Number)
                const fecha = new Date(year, month - 1, day)
                const hoy = new Date(yearHoy, monthHoy - 1, dayHoy)
                const diasRestantes = Math.ceil((fecha.getTime() - hoy.getTime()) / (1000 * 60 * 60 * 24))
                return diasRestantes <= 7 && diasRestantes > 0 && fechaPagoStr >= hoyStr
              })()
              
              let estadoTexto: string = pago.estado
              let estadoColor = 'bg-gray-100 text-gray-800'
              let borderColor = 'border-gray-200'
              
              if (pago.estado === 'completado') {
                estadoTexto = 'Completado'
                estadoColor = 'bg-green-100 text-green-800'
                borderColor = 'border-green-300 border-l-4'
              } else if (pago.estado === 'cancelado') {
                estadoTexto = 'Cancelado'
                estadoColor = 'bg-gray-100 text-gray-800'
              } else if (vencido) {
                estadoTexto = 'Vencido'
                estadoColor = 'bg-red-100 text-red-800'
                borderColor = 'border-red-300 border-l-4'
              } else if (porVencer) {
                estadoTexto = 'Por vencer'
                estadoColor = 'bg-yellow-100 text-yellow-800'
                borderColor = 'border-yellow-300 border-l-4'
              } else {
                estadoTexto = 'En curso'
                estadoColor = 'bg-blue-100 text-blue-800'
                borderColor = 'border-blue-300 border-l-4'
              }
              
              return (
                <Link
                  key={pago.id}
                  href={`/pagos/${pago.id}`}
                  className={`block border rounded p-3 hover:opacity-80 transition ${borderColor}`}
                >
                  <div className="flex justify-between items-center">
                    <div>
                      <p className="font-semibold">${pago.monto.toLocaleString()}</p>
                      <p className="text-sm text-gray-600">
                        {format(new Date(pago.fecha_pago), 'dd/MM/yyyy')} - {pago.metodo_pago || 'Sin método'}
                      </p>
                    </div>
                    <span className={`px-2 py-1 rounded text-xs font-medium ${estadoColor}`}>
                      {estadoTexto}
                    </span>
                  </div>
                </Link>
              )
            })}
          </div>
        )}
      </div>

      {/* Recordatorios */}
      {recordatorios.length > 0 && (
        <div className="bg-white p-6 rounded-lg shadow">
          <div className="flex justify-between items-center mb-4">
            <h2 className="text-2xl font-bold">Recordatorios Pendientes</h2>
            <Link
              href={`/recordatorios?cliente_id=${cliente.id}`}
              className="text-blue-600 hover:underline text-sm"
            >
              Ver todos →
            </Link>
          </div>
          <div className="space-y-3">
            {recordatorios.map((recordatorio) => {
              const fechaRecordatorio = new Date(recordatorio.fecha_recordatorio)
              const esPasado = fechaRecordatorio < new Date()
              return (
                <div
                  key={recordatorio.id}
                  className={`border rounded p-3 ${esPasado ? 'border-l-4 border-red-500' : ''}`}
                >
                  <p className="font-semibold">{recordatorio.titulo}</p>
                  {recordatorio.descripcion && (
                    <p className="text-sm text-gray-600">{recordatorio.descripcion}</p>
                  )}
                  <p className="text-xs text-gray-500 mt-1">
                    {format(fechaRecordatorio, 'dd/MM/yyyy HH:mm')}
                    {recordatorio.tipo && ` - ${recordatorio.tipo}`}
                  </p>
                </div>
              )
            })}
          </div>
        </div>
      )}
    </div>
  )
}
